<?php

/**
 * Plugin Name: Kingpost Wordpress
 * Description: Plugin de sincronização Kingpost
 * Version:           1.4.0
 * Author:            Kingpost
 * Author URI:
 * License:           GPL v2
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Update URI:        https://send.kingpost.com.br/wp-plugin/info.json
 */

if (defined('WPINC')) {

    class Publifacil
    {

        public function init()
        {
            require_once 'analytcs.php';
            $analytics = new Analytics();

            add_action('rest_api_init', [$this, 'set_endpoints']);

            add_action('admin_init', [$this, 'set_config']);

            add_action('template_redirect', [$this, 'is_maintenance']);

            $update = new PublifacilUpdater();
        }
        public function is_maintenance()
        {
            global $wp;
            if ($wp->request == 'manutencao') {
                return;
            }
            if (true == get_option("publifacil_maintenance_set") || "1" == get_option("publifacil_maintenance_set")) {
                wp_redirect('/manutencao', 301);
                exit();
            }
        }

        public function set_config()
        {
            register_setting('general', 'publifacil_api_key');
        }

        public function curl_request($url)
        {

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
            curl_setopt($ch, CURLOPT_URL, $url);
            $result = curl_exec($ch);
            curl_close($ch);
            return (json_decode($result, true));
        }




        public function meta_dados($data)
        {
            $params = $data->get_params();
            if (!$params['api_key']) {
                return new WP_REST_Response('Parametros inválidos', 400);
            }

            require_once(ABSPATH . 'wp-admin/includes/class-wp-debug-data.php');
            \WP_Debug_Data::check_for_updates();

            $qtd_comments = count(get_comments());
            $themes_update = $this->get_themes_data(true)['updates_pending'];
            $plugin_update = $this->get_plugins_data(true)['updates_pending'];

            $args = array(
                'numberposts'   => -1,
                'post_type'     => 'post',
                'post_status'   => array('draft'),
            );
            $total_drafts = count(get_posts($args));


            $wp_get_all_authors_and_categories = $this->wp_get_all_authors_and_categories();
            return [
                'qtd_comments' => $qtd_comments,
                'themes_update' => $themes_update,
                'plugin_update' => $plugin_update,
                'total_drafts' => $total_drafts,
                'authors' => $wp_get_all_authors_and_categories['authors'],
                'categories' => $wp_get_all_authors_and_categories['categories'],
            ];
        }


        function wp_get_all_authors_and_categories() {
            // Get all authors (users who have published posts)
            $authors = get_users(array(
                'who' => 'authors',
                'has_published_posts' => true,
                'orderby' => 'display_name',
                'order' => 'ASC',
            ));

            $authors_data = array_map(function($author) {
                return array(
                    // 'id'    => $author->ID,
                    'name'  => $author->display_name,
                    'email' => $author->user_email,
                    // 'url'   => get_author_posts_url($author->ID),
                );
            }, $authors);

            // Get all categories
            $categories = get_categories(array(
                'hide_empty' => false,
                'orderby' => 'name',
                'order' => 'ASC',
            ));

            $categories_data = array_map(function($cat) {
                return array(
                    // 'id'   => $cat->term_id,
                    'name' => $cat->name,
                    'slug' => $cat->slug,
                    // 'url'  => get_category_link($cat->term_id),
                );
            }, $categories);

            return array(
                'authors' => $authors_data,
                'categories' => $categories_data,
            );
        }


        public function set_endpoints()
        {
            //Endpoints Settings
            register_rest_route('publifacil', 'remover_instalacao', array(
                'methods' => 'post',
                'callback' => array($this, 'remover_instalacao')
            ));

            register_rest_route('publifacil', 'validar', array(
                'methods' => 'post',
                'callback' => array($this, 'validar_instalacao')
            ));

            register_rest_route('publifacil', 'postar', array(
                'methods' => 'post',
                'callback' => array($this, 'post_recebido')
            ));

            register_rest_route('publifacil', 'alterar_post', array(
                'methods' => 'post',
                'callback' => array($this, 'alterar_post_recebido')
            ));

            register_rest_route('publifacil', 'excluir_post', array(
                'methods' => 'post',
                'callback' => array($this, 'excluir_post')
            ));

            register_rest_route('publifacil', 'search_posts', array(
                'methods' => 'post',
                'callback' => array($this, 'search_posts')
            ));

            register_rest_route('publifacil', 'get_post', array(
                'methods' => 'post',
                'callback' => array($this, 'get_post')
            ));

            register_rest_route('publifacil', 'metadados', array(
                'methods' => 'post',
                'callback' => array($this, 'meta_dados')
            ));

            register_rest_route('publifacil', 'get_comments', array(
                'methods' => 'post',
                'callback' => array($this, 'get_comments')
            ));

            register_rest_route('publifacil', 'edit_comment', array(
                'methods' => 'post',
                'callback' => array($this, 'edit_comment')
            ));

            register_rest_route('publifacil', 'change_maintenance', array(
                'methods' => 'post',
                'callback' => array($this, 'change_maintenance')
            ));

            register_rest_route('publifacil', 'get_plugins_data', array(
                'methods' => 'post',
                'callback' => array($this, 'get_plugins_data')
            ));

            register_rest_route('publifacil', 'update_plugin', array(
                'methods' => 'post',
                'callback' => array($this, 'update_plugin')
            ));

            register_rest_route('publifacil', 'update_plugin', array(
                'methods' => 'get',
                'callback' => array($this, 'update_plugin')
            ));

            register_rest_route('publifacil', 'get_themes_data', array(
                'methods' => 'get',
                'callback' => array($this, 'get_themes_data')
            ));

            register_rest_route('publifacil', 'get_themes_data', array(
                'methods' => 'post',
                'callback' => array($this, 'get_themes_data')
            ));

            register_rest_route('publifacil', 'update_theme', array(
                'methods' => 'post',
                'callback' => array($this, 'update_theme')
            ));

            register_rest_route('publifacil', 'update_theme', array(
                'methods' => 'get',
                'callback' => array($this, 'update_theme')
            ));

            register_rest_route('publifacil', 'get_posts_views_count', array(
                'methods' => 'get',
                'callback' => array($this, 'get_posts_views_count')
            ));

            register_rest_route('publifacil', 'get_posts_views_count', array(
                'methods' => 'post',
                'callback' => array($this, 'get_posts_views_count')
            ));

            register_rest_route('publifacil', 'sincronizar_autores', array(
                'methods' => 'post',
                'callback' => array($this, 'sincronizar_autores')
            ));
        }

        public function sincronizar_autores($autores)
        {
            foreach ($autores as $author) {
                $existing_author = get_user_by('name', $author['name']);
                if (!$existing_author) {
                    $user_id = wp_create_user($author['name'], wp_generate_password(), $author['email']);
                    if (!is_wp_error($user_id)) {
                        wp_update_user(array(
                            'ID' => $user_id,
                            'display_name' => $author['name'],
                            'role' => 'author'
                        ));
                    }
                }
                //Update author email if exists
                else {

                    //check if existing author have @kingpost.com.br email
                    if (strpos($existing_author->user_email, '@kingpost.com.br') === false) {
                        continue;
                    }
                    
                    wp_update_user(array(
                        'ID' => $existing_author->ID,
                        'user_email' => $author['email']
                    ));
                }
            }
        }


        function get_posts_views_count($data)
        {
            $post_ids = $data['post_ids'];
            $meta_key = 'post_views_count';
            global $wpdb;

            if (empty($post_ids)) {
                return [];
            }

            $placeholders = implode(',', array_fill(0, count($post_ids), '%d'));

            $results = $wpdb->get_results(
                $wpdb->prepare(
                    "
                    SELECT post_id, meta_value AS views
                    FROM $wpdb->postmeta
                    WHERE post_id IN ($placeholders)
                    AND meta_key = %s",
                    array_merge($post_ids, [$meta_key])
                ),
                OBJECT_K
            );

            $views = [];
            foreach ($post_ids as $id) {
                $views[$id] = isset($results[$id]) ? intval($results[$id]->views) : 0;
            }
            return new WP_REST_Response($views);
        }

        public function get_themes_data($interno = true)
        {
            $updates = get_site_transient('update_themes');
            $allThemes = wp_get_themes();


            $totalWithUpgrades = 0;

            foreach ($allThemes as $key => $theme) {
                $themeJson =  json_encode((array)(($theme)));

                $themeArray =  json_decode($themeJson, true);

                $cleaning = array_shift($themeArray);

                $cleaning = array_shift($themeArray);

                $themeHeaders = array_shift($themeArray);

                $slug  = $themeHeaders["TextDomain"];

                $pluginData = [];

                $updateData = ($updates->response && isset($updates->response[$slug])  ? $updates->response[$slug] : null);

                $currentPlugin = $themeHeaders ?? [];
                if ($updateData && version_compare($currentPlugin["Version"], $updateData["new_version"] ?? 0, '<')) {
                    $pluginData = [
                        "theme" => $currentPlugin,
                        "slug" => $slug,
                        "have_update" => true,
                        "update_data" => $updateData
                    ];
                    $totalWithUpgrades++;
                } else {
                    $pluginData = [
                        "theme" => $currentPlugin,
                        "slug" => $slug,
                        "have_update" => false,
                        "update_data" => $updateData
                    ];
                }
                $plugins[] = $pluginData;
            }

            $response = [
                "themes" => $plugins,
                "updates_pending" => $totalWithUpgrades
            ];

            if ($interno === true) {
                return $response;
            }

            return new WP_REST_Response($response);
        }


        public function get_plugins_data($interno = true)
        {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
            $updates = get_site_transient('update_plugins');
            $allPlugins = get_plugins();
            $active  = get_option('active_plugins', array());
            $plugins = [];
            $totalWithUpgrades = 0;
            if (!isset($updates->response)) {
                return new WP_REST_Response("Todos os plugins estão atualizados");
            }
            foreach ($active as $key => $plugin_file) {

                $pluginData = [];
                $updateData = (array) ($updates->response ? $updates->response[$plugin_file] : null);
                $currentPlugin = $allPlugins[$plugin_file] ?? [];
                if (version_compare($currentPlugin["Version"], $updateData["new_version"], '<')) {
                    $pluginData = [
                        "plugin" => $currentPlugin,
                        "slug" => $plugin_file,
                        "have_update" => true,
                        "update_data" => $updateData
                    ];
                    $totalWithUpgrades++;
                } else {
                    $pluginData = [
                        "plugin" => $currentPlugin,
                        "slug" => $plugin_file,
                        "have_update" => false,
                        "update_data" => $updateData
                    ];
                }
                $plugins[] = $pluginData;
            }

            $response = [
                "plugins" => $plugins,
                "updates_pending" => $totalWithUpgrades
            ];

            if ($interno === true) {
                return $response;
            }

            return new WP_REST_Response($response);
        }

        public function update_plugin($data)
        {
            $params = $data->get_params();

            ob_start();
            try {
                $plugin_slug = $params["plugin"];

                require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
                require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php';
                require_once ABSPATH . 'wp-admin/includes/file.php';
                require_once ABSPATH . 'wp-admin/includes/misc.php';
                require_once ABSPATH . 'wp-load.php';

                $upgrader = new KingpostPluginsUpdater(new WP_Upgrader_Skin());

                $result = $upgrader->upgrade($plugin_slug, array(
                    'destination' => $plugin_slug,
                    'clear_destination' => false,
                    'clear_working' => false
                ));
                if (is_wp_error($result)) {
                    return false;
                }
                ob_end_clean();
                return $this->get_plugins_data();
            } catch (\Throwable $th) {
                ob_end_clean();
                return new WP_REST_Response('Ocorreu um erro', 400);
            }
        }


        public function update_theme($data)
        {
            $params = $data->get_params();

            try {
                $theme = $params["theme"];

                require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
                require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php';
                require_once ABSPATH . 'wp-admin/includes/file.php';
                require_once ABSPATH . 'wp-admin/includes/misc.php';
                require_once ABSPATH . 'wp-load.php';

                $upgrader = new Theme_Upgrader(new WP_Upgrader_Skin());

                $result = $upgrader->upgrade($theme);
                if (is_wp_error($result)) {
                    die($result);
                    return false;
                }
                return true;
            } catch (\Throwable $th) {
                die($th->getMessage());
                return new WP_REST_Response('Ocorreu um erro', 400);
            }
        }


        public function change_maintenance($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }
            if (!update_option('publifacil_maintenance_set', (bool)($params["manutencao"]))) {
                add_option('publifacil_maintenance_set', (bool)($params["manutencao"]));
            }
            return new WP_REST_Response(false);
        }

        public function get_comments($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            return new WP_REST_Response(get_comments());
        }

        public function edit_comment($data)
        {
            $params = $data->get_params();
            $comment_id = $params["id"];
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            if ($params["comment_approved"] == 3) {
                wp_delete_comment($comment_id);
                return new WP_REST_Response(get_comments());
            }
            $commentarr = array();
            $commentarr['comment_ID'] = $comment_id;
            $commentarr['comment_approved'] = $params["comment_approved"] == 1 ? 1 : 0;
            wp_update_comment($commentarr);

            return new WP_REST_Response(get_comments());
        }





        public function get_post($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            $post = get_post($params['post_id'], ARRAY_A);
            if (count($post['post_category']) > 0) {
                $cats = $post['post_category'];
                $post['cat'] = array_map(function ($user) {
                    return (get_category($user))->name;
                }, $cats);
            }

            if (isset($post['post_author'])) {
                $author = get_the_author_meta('display_name', $post['post_author']);
                $post['author'] = $author;
            }


            $post['post_format'] = get_post_format($post['ID']) ?: '';
            $post['imagem_destaque'] = get_the_post_thumbnail_url($post['ID']);
            return new WP_REST_Response($post);
        }

        public function search_posts($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            $query = new WP_Query(array(
                's' => $params['s'],
                'post_type' => 'any',
                'suppress_filters' => TRUE,
                'posts_per_page' => '-1'
            ));

            $result = (array) $query;
            $posts = array_map(fn($post) => ['ID' => $post->ID, 'post_title' => $post->post_title, 'post_date' => $post->post_date, 'post_type' => $post->post_type, 'url' => get_permalink($post->ID)], $result['posts']);
            return new WP_REST_Response($posts);
        }

        public function remover_instalacao($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            delete_option('publifacil_api_key');

            return new WP_REST_Response('Instalação desativada');
        }


        public function validar_instalacao($data)
        {
            $params = $data->get_params();
            if (!$params['api_key']) {
                return new WP_REST_Response('Parametros inválidos', 400);
            }

            $settings = [
                'api_key' => $params['api_key']
            ];

            $current_settings = $this->get_settings();
            if ($current_settings['api_key'] != false) {
                if ($current_settings && $current_settings['api_key'] != $settings['api_key']) //Ja possui configuração porém tem valor diferente da configuração
                {
                    return new WP_REST_Response('Instalação com chave incorreta', 400);
                }

                if ($current_settings && $current_settings['api_key'] == $settings['api_key']) //Ja possui configuração
                {
                    return new WP_REST_Response('Instalação previamente validada');
                }
            }


            if (!update_option('publifacil_api_key', $settings['api_key'])) {
                add_option('publifacil_api_key', $settings['api_key']);
            }
            try {
                $metadados = $this->meta_dados($data);
            } catch (\Throwable $th) {
                $metadados = null;
            }

            return new WP_REST_Response(['message' => 'Instalação Validada', 'metadados' => $metadados]);
        }

        public function get_settings()
        {
            $api_key = get_option('publifacil_api_key');
            return [
                'api_key' => $api_key
            ];
        }

        public function validate_api_key($api_key)
        {
            $settings = $this->get_settings();

            if (!$settings) {
                return false;
            }

            if (empty($settings['api_key'])) {
                if (!update_option('publifacil_api_key', $api_key)) {
                    add_option('publifacil_api_key', $api_key);
                }
                return true;
            }

            if ($settings['api_key'] != $api_key) {
                return false;
            }

            if ($settings['api_key'] == $api_key) {
                return true;
            }

            return false;
        }

        public function excluir_post($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            try {
                wp_delete_post($params['post_id']);
                return new WP_REST_Response('Excluido');
            } catch (\Throwable $th) {
                return new WP_REST_Response('Ocorreu um erro', 400);
            }
        }
        public function post_recebido($data)
        {
            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            $author = $this->get_create_user($params['author'], $params['author_email'] ?? null);
            $category_id = null;
            if (isset($params['post_category']) && !empty($params['post_category'])) {

                if (is_array($params['post_category'])) {

                    foreach ($params['post_category'] as $key => $value) {
                        $category_id[] = $this->get_create_category($value);
                    }
                } else {
                    $category_id[] = $this->get_create_category($params['post_category']);
                }
            }

            $content = $this->sanitize_content($params['content']);

            $new_post = [
                'post_author' => $author->id,
                'post_content' => $content,
                'post_title' => $params['title'],


                'post_category' => $category_id,
                'post_type' => $params['post_type'],
                'post_status' => $params['post_status'],
                'post_date' => $params['post_date'],
                'tags_input' => $params['tags_input'],
                'post_excerpt' => $params['post_excerpt'],

            ];


            $post_id = wp_insert_post($new_post);
            if (isset($params['post_format']) && !empty($params['post_format'])) {
                set_post_format($post_id, $params['post_format']);
            }

            if (isset($params['post_image']) && !empty($params['post_image'])) {
                $image_id = $this->upload_image_from_url($params['post_image'], $params['post_image_title'] ?? '');
                if ($image_id) {
                    set_post_thumbnail($post_id, $image_id);
                }
            }


            return new WP_REST_Response(['url' => get_permalink($post_id), 'wp_post_id' => $post_id, 'wp_post_url' => get_permalink($post_id), 'wp_post_date' => $params['post_date']]);
        }

        public function alterar_post_recebido($data)
        {
            require_once(ABSPATH . 'wp-admin/includes/post.php');
            require_once(ABSPATH . 'wp-admin/includes/user.php');
            require_once(ABSPATH . 'wp-includes/pluggable.php');

            $params = $data->get_params();
            if ($this->validate_api_key($params['api_key']) !== true) {
                return new WP_REST_Response('Não autorizado', 400);
            }

            $get_admin = get_users(['role' => 'administrator']);
            if (empty($get_admin)) {
                $user_data = array(
                    'user_pass' => bin2hex(random_bytes(10)),
                    'user_login' => 'Admin Kingpost',
                    'user_email' => bin2hex(random_bytes(10)) . '@kingpost.com.br',
                    'role' => 'administrator',
                );
                $id = wp_insert_user($user_data);
                wp_set_current_user($id);
            } else {
                wp_set_current_user($get_admin[0]->data->ID);
            }



            $author = $this->get_create_user($params['author'], $params['author_email'] ?? null);
            // if (isset($params['post_category']) && !empty($params['post_category'])) {
            //     $category_id = $this->get_create_category($params['post_category']);
            // } else {
            //     $category_id = null;
            // }


            if (isset($params['post_category']) && !empty($params['post_category'])) {

                if (is_array($params['post_category'])) {

                    foreach ($params['post_category'] as $key => $value) {
                        $category_id[] = $this->get_create_category($value);
                    }
                } else {
                    $category_id[] = $this->get_create_category($params['post_category']);
                }
            } else {
                $category_id = null;
            }

            $content = $this->sanitize_content($params['content']);

            $new_post = [
                'post_ID' => $params['post_id'],
                'post_author' => $author->id,
                'post_content' => $content,
                'post_title' => $params['title'],
                'post_category' => $category_id,
                'post_type' => $params['post_type'],
                'post_status' => $params['post_status'],
                'post_date' => $params['post_date'],
                'tags_input' => $params['tags_input'],
                'post_excerpt' => $params['post_excerpt'],

            ];


            $post_id = edit_post($new_post);
            if (isset($params['post_format']) && !empty($params['post_format'])) {
                set_post_format($post_id, $params['post_format']);
            }

            if (isset($params['post_image']) && !empty($params['post_image'])) {
                $image_id = $this->upload_image_from_url($params['post_image'], $params['post_image_title'] ?? '');
                if ($image_id) {
                    set_post_thumbnail($post_id, $image_id);
                }
            }


            return new WP_REST_Response(['url' => get_permalink($post_id), 'wp_post_id' => $post_id, 'wp_post_url' => get_permalink($post_id), 'wp_post_date' => $params['post_date']]);
        }


        public function sanitize_content($content)
        {
            $rgx = '/<img(.?){1,}src(.?){1,}"(.?){1,}>/U';
            $content = stripslashes($content);
            $new_content = preg_replace_callback($rgx, function ($matches) {
                $match = $matches[0];
                preg_match('/src="(.?){0,}"/U', $match, $item);
                $url = str_replace("src=", "", ($item[0]));
                $url = str_replace('"', '', $url);
                $image_id = $this->upload_image_from_url($url);
                if ($image_id) {
                    $image_url = wp_get_attachment_image_url($image_id, 'full');
                    return  '<img src="' . $image_url . '">';
                } else {
                    return $match;
                }
            }, $content);
            return $new_content;
        }

        function upload_image_from_url($image_url, $image_name = '')
        {
            require_once(ABSPATH . 'wp-admin/includes/file.php');

            $temp_file = download_url($image_url, 300, false);

            if (is_wp_error($temp_file)) {

                return false;
            }

            $file = array(
                'name'     => basename($image_url),
                'type'     => mime_content_type($temp_file),
                'tmp_name' => $temp_file,
                'size'     => filesize($temp_file),
            );
            $sideload = wp_handle_sideload(
                $file,
                array(
                    'test_form'   => false
                )
            );

            if (!empty($sideload['error'])) {
                return false;
            }

            $attachment_id = wp_insert_attachment(
                array(
                    'guid'           => $sideload['url'],
                    'post_mime_type' => $sideload['type'],
                    'post_title'     => $image_name ?? basename($sideload['file']),
                    'post_content'   => '',
                    'post_status'    => 'inherit',
                ),
                $sideload['file']
            );

            if (is_wp_error($attachment_id) || !$attachment_id) {
                return false;
            }

            require_once(ABSPATH . 'wp-admin/includes/image.php');
            wp_update_attachment_metadata(
                $attachment_id,
                wp_generate_attachment_metadata($attachment_id, $sideload['file'])
            );

            return $attachment_id;
        }


        private function get_create_category($category_name)
        {
            $cats = get_terms([
                'fields' => 'ids',
                'taxonomy' => 'category',
                'slug' => remove_accents($category_name),
                'hide_empty' => false,
            ]);

            $cat = array_pop($cats);
            if (!$cat) {
                $new_cat = wp_insert_term(
                    $category_name,
                    'category',
                    array(
                        'slug' => remove_accents($category_name),
                    )
                );
                return $new_cat['term_id'];
            }
            return $cat;
        }

        private function get_create_user($user_name_id, $email = null)
        {

            $user = get_user_by('ID', $user_name_id);
            if ($user) {
                return $user;
            }

            $user = get_user_by('login', $user_name_id);
            if ($user) {
                return $user;
            }

            $user = get_user_by('slug', $user_name_id);
            if ($user) {
                return $user;
            }

            $user = get_user_by('email', $user_name_id);
            if ($user) {
                return $user;
            }

            $user = get_user_by('email', $email);
            if ($user) {
                return $user;
            }

            //Cadastra Usuario
            $user_data = array(
                'user_pass' => null,
                'user_login' => null,
                'user_email' => null,
                'role' => 'author',
            );
            if (is_int($user_name_id)) {
                $user_data['ID'] = $user_name_id;
                $user_data['user_login'] = 'Usuário Kingpost  API - ' . bin2hex(random_bytes(2));
            } else {
                $user_data['user_login'] = $user_name_id;
            }
            $user_data['user_pass'] = bin2hex(random_bytes(10));
            $user_data['user_email'] =  $email ?? bin2hex(random_bytes(10)) . '@kingpost.com.br';
            $id = wp_insert_user($user_data);
            $user = get_user_by('ID', $id);
            return $user;
        }
    }


    function dd($mix)
    {
        echo '<pre>';
        print_r($mix);
        die();
    }

    if (!class_exists('PublifacilUpdater')) {

        class PublifacilUpdater
        {

            public $plugin_slug;
            public $version;
            public $cache_key;
            public $cache_allowed;

            public function __construct()
            {

                $this->plugin_slug = 'kingpost-client/kingpost.php';
                $this->version = '1.4.0';
                $this->cache_key = 'custom_upd';
                $this->cache_allowed = true;

                add_filter('plugins_api', array($this, 'info'), 20, 3);
                add_filter('site_transient_update_plugins', array($this, 'update'));
                add_action('upgrader_process_complete', array($this, 'purge'), 10, 2);
            }





            public function request()
            {

                $remote = get_transient($this->cache_key);

                if (false === $remote || !$this->cache_allowed) {

                    $remote = wp_remote_get(
                        'https://send.kingpost.com.br/wp-plugin/info.json',
                        array(
                            'timeout' => 10,
                            'headers' => array(
                                'Accept' => 'application/json'
                            )
                        )
                    );

                    if (
                        is_wp_error($remote)
                        || 200 !== wp_remote_retrieve_response_code($remote)
                        || empty(wp_remote_retrieve_body($remote))
                    ) {
                        return false;
                    }

                    set_transient($this->cache_key, $remote, DAY_IN_SECONDS);
                }

                $remote = json_decode(wp_remote_retrieve_body($remote));

                return $remote;
            }


            function info($res, $action, $args)
            {

                // print_r( $action );
                // print_r( $args );

                // do nothing if you're not getting plugin information right now
                if ('plugin_information' !== $action) {
                    return $res;
                }

                // do nothing if it is not our plugin
                if ($this->plugin_slug !== $args->slug) {
                    return $res;
                }

                // get updates
                $remote = $this->request();

                if (!$remote) {
                    return $res;
                }

                $res = new stdClass();

                $res->name = $remote->name;
                $res->slug = $remote->slug;
                $res->version = $remote->version;
                $res->tested = $remote->tested;
                $res->requires = $remote->requires;
                $res->author = $remote->author;
                $res->author_profile = $remote->author_profile;
                $res->download_link = $remote->download_url;
                $res->trunk = $remote->download_url;
                $res->requires_php = $remote->requires_php;
                $res->last_updated = $remote->last_updated;

                $res->sections = array(
                    'description' => $remote->sections->description,
                    'installation' => $remote->sections->installation,
                    'changelog' => $remote->sections->changelog
                );

                if (!empty($remote->banners)) {
                    $res->banners = array(
                        'low' => $remote->banners->low,
                        'high' => $remote->banners->high
                    );
                }

                return $res;
            }

            public function update($transient)
            {

                if (empty($transient->checked)) {
                    return $transient;
                }

                $remote = $this->request();
                if (
                    $remote
                    && version_compare($this->version, $remote->version, '<')
                    && version_compare($remote->requires, get_bloginfo('version'), '<=')
                    && version_compare($remote->requires_php, PHP_VERSION, '<')
                ) {
                    $res = new stdClass();
                    $res->slug = $this->plugin_slug;
                    $res->plugin = 'kingpost-client/kingpost.php';
                    $res->new_version = $remote->version;
                    $res->tested = $remote->tested;
                    $res->package = $remote->download_url;

                    $transient->response[$res->plugin] = $res;
                }
                return $transient;
            }

            public function purge($upgrader, $options)
            {

                if (
                    $this->cache_allowed
                    && 'update' === $options['action']
                    && 'plugin' === $options['type']
                ) {
                    // just clean the cache when new plugin version is installed
                    delete_transient($this->cache_key);
                }
            }
        }
    }


    if (!class_exists('KingpostPluginsUpdater')) {
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        require_once ABSPATH . 'wp-load.php';

        class KingpostPluginsUpdater extends Plugin_Upgrader
        {
            public function run($options)
            {

                $defaults = array(
                    'package'                     => '', // Please always pass this.
                    'destination'                 => '', // ...and this.
                    'clear_destination'           => false,
                    'clear_working'               => true,
                    'abort_if_destination_exists' => true, // Abort if the destination directory exists. Pass clear_destination as false please.
                    'is_multi'                    => false,
                    'hook_extra'                  => array(), // Pass any extra $hook_extra args here, this will be passed to any hooked filters.
                );

                $options = wp_parse_args($options, $defaults);
                $options = apply_filters('upgrader_package_options', $options);

                if (!$options['is_multi']) { // Call $this->header separately if running multiple times.
                    $this->skin->header();
                }

                // Connect to the filesystem first.
                $res = $this->fs_connect(array(WP_CONTENT_DIR, $options['destination']));
                // Mainly for non-connected filesystem.
                if (!$res) {
                    if (!$options['is_multi']) {
                        $this->skin->footer();
                    }
                    return false;
                }

                $this->skin->before();

                if (is_wp_error($res)) {
                    $this->skin->error($res);
                    $this->skin->after();
                    if (!$options['is_multi']) {
                        $this->skin->footer();
                    }
                    return $res;
                }

                $download = $this->download_package($options['package'], false, $options['hook_extra']);

                // if ( is_wp_error( $download ) && $download->get_error_data( 'softfail-filename' ) ) {

                //     if ( 'signature_verification_no_signature' !== $download->get_error_code() || WP_DEBUG ) {
                //         $this->skin->feedback( $download->get_error_message() );

                //         wp_version_check(
                //             array(
                //                 'signature_failure_code' => $download->get_error_code(),
                //                 'signature_failure_data' => $download->get_error_data(),
                //             )
                //         );
                //     }

                //     $download = $download->get_error_data( 'softfail-filename' );
                // }

                if (is_wp_error($download)) {

                    $this->skin->error($download);
                    $this->skin->after();
                    if (!$options['is_multi']) {
                        $this->skin->footer();
                    }
                    return $download;
                }

                $delete_package = ($download !== $options['package']);

                $working_dir = $this->unpack_package($download, $delete_package);
                if (is_wp_error($working_dir)) {
                    $this->skin->error($working_dir);
                    $this->skin->after();
                    if (!$options['is_multi']) {
                        $this->skin->footer();
                    }
                    return $working_dir;
                }
                $result = $this->install_package(
                    array(
                        'source'                      => $working_dir,
                        'destination'                 => $options['destination'],
                        'clear_destination'           => false,
                        'abort_if_destination_exists' => false,
                        'clear_working'               => false,
                        'hook_extra'                  => $options['hook_extra'],
                    )
                );
                $result = apply_filters('upgrader_install_package_result', $result, $options['hook_extra']);

                $this->skin->set_result($result);

                if (is_wp_error($result)) {
                    if (!empty($options['hook_extra']['temp_backup'])) {
                        $this->temp_restores[] = $options['hook_extra']['temp_backup'];
                        add_action('shutdown', array($this, 'restore_temp_backup'));
                    }
                    $this->skin->error($result);

                    if (!method_exists($this->skin, 'hide_process_failed') || !$this->skin->hide_process_failed($result)) {
                        $this->skin->feedback('process_failed');
                    }
                } else {
                    $this->skin->feedback('process_success');
                }

                $this->skin->after();

                if (!empty($options['hook_extra']['temp_backup'])) {
                    add_action('shutdown', array($this, 'delete_temp_backup'), 100, 0);
                }

                if (!$options['is_multi']) {
                    do_action('upgrader_process_complete', $this, $options['hook_extra']);

                    $this->skin->footer();
                }

                return $result;
            }

            public function install_package($args = array())
            {
                global $wp_filesystem, $wp_theme_directories;

                $defaults = array(
                    'source'                      => '', // Please always pass this.
                    'destination'                 => '', // ...and this.
                    'clear_destination'           => false,
                    'clear_working'               => false,
                    'abort_if_destination_exists' => true,
                    'hook_extra'                  => array(),
                );

                $args = wp_parse_args($args, $defaults);

                // These were previously extract()'d.
                $source            = $args['source'];
                $destination       = $args['destination'];
                $clear_destination = $args['clear_destination'];

                if (function_exists('set_time_limit')) {
                    set_time_limit(300);
                }


                if (empty($source) || empty($destination)) {
                    return new WP_Error('bad_request', $this->strings['bad_request']);
                }
                $this->skin->feedback('installing_package');

                /**
                 * Filters the installation response before the installation has started.
                 *
                 * Returning a value that could be evaluated as a `WP_Error` will effectively
                 * short-circuit the installation, returning that value instead.
                 *
                 * @since 2.8.0
                 *
                 * @param bool|WP_Error $response   Installation response.
                 * @param array         $hook_extra Extra arguments passed to hooked filters.
                 */

                // Retain the original source and destinations.
                $remote_source     = $args['source'];
                $local_destination = $destination;

                $source_files       = array_keys($wp_filesystem->dirlist($remote_source));
                $remote_destination = $wp_filesystem->find_folder($local_destination);

                // Locate which directory to copy to the new folder. This is based on the actual folder holding the files.
                if (1 === count($source_files) && $wp_filesystem->is_dir(trailingslashit($args['source']) . $source_files[0] . '/')) {
                    // Only one folder? Then we want its contents.
                    $source = trailingslashit($args['source']) . trailingslashit($source_files[0]);
                } elseif (0 === count($source_files)) {
                    // There are no files?
                    return new WP_Error('incompatible_archive_empty', $this->strings['incompatible_archive'], $this->strings['no_files']);
                } else {
                    /*
                     * It's only a single file, the upgrader will use the folder name of this file as the destination folder.
                     * Folder name is based on zip filename.
                     */
                    $source = trailingslashit($args['source']);
                }


                /**
                 * Filters the source file location for the upgrade package.
                 *
                 * @since 2.8.0
                 * @since 4.4.0 The $hook_extra parameter became available.
                 *
                 * @param string      $source        File source location.
                 * @param string      $remote_source Remote file source location.
                 * @param WP_Upgrader $upgrader      WP_Upgrader instance.
                 * @param array       $hook_extra    Extra arguments passed to hooked filters.
                 */
                $source = apply_filters('upgrader_source_selection', $source, $remote_source, $this, $args['hook_extra']);

                if (is_wp_error($source)) {
                    return $source;
                }

                if (!empty($args['hook_extra']['temp_backup'])) {
                    $temp_backup = $this->move_to_temp_backup_dir($args['hook_extra']['temp_backup']);

                    if (is_wp_error($temp_backup)) {
                        return $temp_backup;
                    }

                    $this->temp_backups[] = $args['hook_extra']['temp_backup'];
                }

                // Has the source location changed? If so, we need a new source_files list.
                if ($source !== $remote_source) {
                    $source_files = array_keys($wp_filesystem->dirlist($source));
                }

                /*
                 * Protection against deleting files in any important base directories.
                 * Theme_Upgrader & Plugin_Upgrader also trigger this, as they pass the
                 * destination directory (WP_PLUGIN_DIR / wp-content/themes) intending
                 * to copy the directory into the directory, whilst they pass the source
                 * as the actual files to copy.
                 */
                $protected_directories = array(ABSPATH, WP_CONTENT_DIR, WP_PLUGIN_DIR, WP_CONTENT_DIR . '/themes');

                if (is_array($wp_theme_directories)) {
                    $protected_directories = array_merge($protected_directories, $wp_theme_directories);
                }

                if (in_array($destination, $protected_directories, true)) {
                    $remote_destination = trailingslashit($remote_destination) . trailingslashit(basename($source));
                    $destination        = trailingslashit($destination) . trailingslashit(basename($source));
                }

                if ($clear_destination) {
                    // We're going to clear the destination if there's something there.
                    $this->skin->feedback('remove_old');

                    $removed = $this->clear_destination($remote_destination);

                    /**
                     * Filters whether the upgrader cleared the destination.
                     *
                     * @since 2.8.0
                     *
                     * @param true|WP_Error $removed            Whether the destination was cleared.
                     *                                          True upon success, WP_Error on failure.
                     * @param string        $local_destination  The local package destination.
                     * @param string        $remote_destination The remote package destination.
                     * @param array         $hook_extra         Extra arguments passed to hooked filters.
                     */
                    $removed = apply_filters('upgrader_clear_destination', $removed, $local_destination, $remote_destination, $args['hook_extra']);

                    if (is_wp_error($removed)) {
                        return $removed;
                    }
                } elseif ($args['abort_if_destination_exists'] && $wp_filesystem->exists($remote_destination)) {
                    /*
                     * If we're not clearing the destination folder and something exists there already, bail.
                     * But first check to see if there are actually any files in the folder.
                     */
                    $_files = $wp_filesystem->dirlist($remote_destination);
                    if (!empty($_files)) {
                        $wp_filesystem->delete($remote_source, true); // Clear out the source files.
                        return new WP_Error('folder_exists', $this->strings['folder_exists'], $remote_destination);
                    }
                }

                /*
                 * If 'clear_working' is false, the source should not be removed, so use copy_dir() instead.
                 *
                 * Partial updates, like language packs, may want to retain the destination.
                 * If the destination exists or has contents, this may be a partial update,
                 * and the destination should not be removed, so use copy_dir() instead.
                 */
                if (
                    $args['clear_working']
                    && (
                        // Destination does not exist or has no contents.
                        !$wp_filesystem->exists($remote_destination)
                        || empty($wp_filesystem->dirlist($remote_destination))
                    )
                ) {
                    $result = move_dir($source, $remote_destination, true);
                } else {
                    // Create destination if needed.
                    if (!$wp_filesystem->exists($remote_destination)) {
                        if (!$wp_filesystem->mkdir($remote_destination, FS_CHMOD_DIR)) {
                            return new WP_Error('mkdir_failed_destination', $this->strings['mkdir_failed'], $remote_destination);
                        }
                    }
                    $result = copy_dir($source, $remote_destination);
                }

                // Clear the working directory?
                if ($args['clear_working']) {
                    $wp_filesystem->delete($remote_source, true);
                }

                if (is_wp_error($result)) {
                    return $result;
                }

                $destination_name = basename(str_replace($local_destination, '', $destination));
                if ('.' === $destination_name) {
                    $destination_name = '';
                }

                $this->result = compact('source', 'source_files', 'destination', 'destination_name', 'local_destination', 'remote_destination', 'clear_destination');

                /**
                 * Filters the installation response after the installation has finished.
                 *
                 * @since 2.8.0
                 *
                 * @param bool  $response   Installation response.
                 * @param array $hook_extra Extra arguments passed to hooked filters.
                 * @param array $result     Installation result data.
                 */
                $res = apply_filters('upgrader_post_install', true, $args['hook_extra'], $this->result);

                if (is_wp_error($res)) {
                    $this->result = $res;
                    return $res;
                }

                // Bombard the calling function will all the info which we've just used.
                return $this->result;
            }
        }
    }

    $publifacil = new Publifacil();
    $publifacil->init();


    /**
     * Track a view for a single post.
     */
    function track_post_views($post_id)
    {
        if (! $post_id) {
            return;
        }

        $meta_key = 'post_views_count';
        $cookie_key = "post_viewed_$post_id";
        $ip = $_SERVER['REMOTE_ADDR']; 
        $ip_transient_key = "post_{$post_id}_$ip";


        if (isset($_COOKIE[$cookie_key])) {
            return; 
        }

        if (get_transient($ip_transient_key)) {
            return;
        }

        $views = get_post_meta($post_id, $meta_key, true);
        $views = $views ? intval($views) + 1 : 1;
        update_post_meta($post_id, $meta_key, $views);


        setcookie($cookie_key, '1', time() + 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);

        set_transient($ip_transient_key, 1, HOUR_IN_SECONDS);
    }

    function track_post_views_on_single()
    {
        if (is_singular('post') || is_page()) {
            global $post;
            if ($post && ! is_preview()) {
                track_post_views($post->ID);
            }
        }
    }
    add_action('wp_head', 'track_post_views_on_single');
}

// https://kingpost.com.br/blog4/wp-json/publifacil/get_posts_views_count?post_ids[]=993&post_ids[]=991
